# Weighting functions for point data averaging
# in accordance to Schrön et.al. 2017, HESS

# Definitions

## Horizontal

WrX <- function(r,x,y){
  
  x00 = 3.7
  a00 = 8735; a01 = 22.689; a02 = 11720; a03 = 0.00978; a04 = 9306; a05 = 0.003632   
  a10 = 2.7925e-002; a11 = 6.6577; a12 = 0.028544; a13 = 0.002455; a14 = 6.851e-005; a15 = 12.2755
  a20 = 247970; a21 = 23.289; a22 = 374655; a23 = 0.00191; a24 = 258552 
  a30 = 5.4818e-002; a31 = 21.032; a32 = 0.6373; a33 = 0.0791; a34 = 5.425e-004 
  
  x0 = x00
  A0 = (a00*(1+a03*x)*exp(-a01*y)+a02*(1+a05*x)-a04*y)
  A1 = ((-a10+a14*x)*exp(-a11*y/(1+a15*y))+a12)*(1+x*a13)
  A2 = (a20*(1+a23*x)*exp(-a21*y)+a22-a24*y)
  A3 = a30*exp(-a31*y)+a32-a33*y+a34*x
  
  return((A0*(exp(-A1*r))+A2*exp(-A3*r))*(1-exp(-x0*r)))
}

WrA <- function(r,x,y){
  
  a00 = 8735; a01 = 22.689; a02 = 11720; a03 = 0.00978; a04 = 9306; a05 = 0.003632   
  a10 = 2.7925e-002; a11 = 6.6577; a12 = 0.028544; a13 = 0.002455; a14 = 6.851e-005; a15 = 12.2755
  a20 = 247970; a21 = 23.289; a22 = 374655; a23 = 0.00191; a24 = 258552 
  a30 = 5.4818e-002; a31 = 21.032; a32 = 0.6373; a33 = 0.0791; a34 = 5.425e-004 
  
  A0 = (a00*(1+a03*x)*exp(-a01*y)+a02*(1+a05*x)-a04*y)
  A1 = ((-a10+a14*x)*exp(-a11*y/(1+a15*y))+a12)*(1+x*a13)
  A2 = (a20*(1+a23*x)*exp(-a21*y)+a22-a24*y)
  A3 = a30*exp(-a31*y)+a32-a33*y+a34*x
  
  return(A0*(exp(-A1*r))+A2*exp(-A3*r))
}


WrB <- function(r,x,y){
  
  b00 = 39006; b01 = 15002337; b02 = 2009.24; b03 = 0.01181; b04 = 3.146; b05 = 16.7417; b06 = 3727
  b10 = 6.031e-005; b11 = 98.5; b12 = 0.0013826
  b20 = 11747; b21 = 55.033; b22 = 4521; b23 = 0.01998; b24 = 0.00604; b25 = 3347.4; b26 = 0.00475 
  b30 = 1.543e-002; b31 = 13.29; b32 = 1.807e-002; b33 = 0.0011; b34 = 8.81e-005; b35 = 0.0405; b36 = 26.74 
  
  B0 = (b00-b01/(b02*y+x-0.13))*(b03-y)*exp(-b04*y)-b05*x*y+b06
  B1 = b10*(x+b11)+b12*y
  B2 = (b20*(1-b26*x)*exp(-b21*y*(1-x*b24))+b22-b25*y)*(2+x*b23)
  B3 = ((-b30+b34*x)*exp(-b31*y/(1+b35*x+b36*y))+b32)*(2+x*b33) 
  
  return(B0*(exp(-B1*r))+B2*exp(-B3*r))
}

## Vertical

D86 <- function(r, bd, y) {
  return(1/bd*(8.321+0.14249*(0.96655+exp(-0.01*r))*(20+y)/(0.0429+y)))
}

Wd <- function(d, r, bd, y){
  return(exp(-2*d/D86(r, bd, y)))
}

## Rescaled distance

rscaled <- function(r, p, Hveg, y) {
  Fp <- 0.4922/(0.86-exp(-p/1013.25))
  Fveg <- 1-0.17*(1-exp(-0.41*Hveg))*(1+exp(-9.25*y))
  return(r * Fp * Fveg )  
}

# Conditions

x <-  0.1 # Air Humidity  from 0.1  to 50    in g/m^3
y <-  0.01 # Soil Moisture from 0.02 to  0.50 in m^3/m^3
p <- 900.0 # air pressure in mbar
Hveg <- 0.0 # vegetation height in m
bd <- 1.46

## Radial distance r [m] from the sensor

d <- seq(0.1,    100, 0.1) # cm
r0 <- seq( 0.1,    1, 0.1) # m
r1 <- seq( 1.1,   50, 0.1) # m
r2 <- seq(50.1,  350, 0.1) # m

# Plot

plot(r0, WrX(rscaled(r0, p, Hveg, y), x, y), xlim=c(0,250), ylim=c(5000,500000), type="l", log="y")
lines(r1, WrA(rscaled(r1, p, Hveg, y),x,y))
lines(r2, WrB(rscaled(r2, p, Hveg, y),x,y))

plot(d, Wd(d, rscaled(1, p, Hveg, y), bd, y), type="l")
abline(v=D86(1, bd, y), lty=2)
