%% License

% Matlab code that accompanies the paper: "Towards a conceptualization of the hydrological processes behind changes
% of young water fraction with elevation: a focus on mountainous alpine catchments"
% submitted to Hydroloy and Earth System Science (HESS) 
% Written by Bettina Schaefli, University of Lausanne and University of Bern, bettina.schaefli@giub.unibe.ch
% 
% Creative Commons Attribution Share Alike 3.0 license
% Original author: bettina.schaefli@giub.unibe.ch
% 
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are
% met:
% 
% * Redistributions of source code must retain the above copyright
%   notice, this list of conditions and the following disclaimer.
% * Redistributions in binary form must reproduce the above copyright
%   notice, this list of conditions and the following disclaimer in
%   the documentation and/or other materials provided with the distribution
% 
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
% POSSIBILITY OF SUCH DAMAGE.

%%  Baseflow separation - Duncan (2019)
% https://www.sciencedirect.com/science/article/pii/S0022169419304858

% Input
% strflow: continuous streamflow measurements at river guages or field flumes
% k: recession constant
% c: constant flow added to the exponential decay component

% Output variables:
% baseq: baseflow

function baseq = baseflowFilter(strflow,k,c)


if nargin<2
    k=0.925;    % a value proposed in the literature (Nathan and McMahon, 1990)
    c=0;        
end

%% Calibration guidelines (Duncan, 2019)

% "1) The master recession should show a step up during significant rain
% (except shortly after a previous event, where event runoff is still
% present), since significant rain should recharge groundwater and
% thus increase baseflow. If it does not, a steeper recession (lower k)
% may be more appropriate.

% 2) The master recession should not lie much below total flow in the
% absence of rain (except shortly after an event, where event runoff is
% still present). If it does, the modelled baseflow is receding more
% slowly than the observed flows. A steeper recession (lower k) may be
% more appropriate. 

% 3) The master recession should not cling tightly to total flow in the
% absence of rain and be continuously held down by the not greater
% than total flow condition. If it does, the modelled recession is
% steeper than that of the observed flows. A flatter recession (higher k)
% may be more appropriate.

% 4) If zero flows may occur, the constant c must be a small negative
% flow, typically a few percent of mean flow at the site. If zero flows
% never occur, c may be initially set to zero. The constant c should be
% calibrated using observed flows close to zero.

% 5)Setting the filter parameter in the second pass equal to the recession
% constant in the first pass appears to be satisfactory in each case
% examined.
%%

masterrecession = strflow;
masterrecession(end) = strflow(end)*0.5;
masterrecession(1) = masterrecession(2);

% first pass, compute a Master recession, single backwards pass

for i = length(strflow):-1:2
    masterrecession(i-1)=min(strflow(i-1),(masterrecession(i)-c)/k+c);
    masterrecession(i-1)=max(0,masterrecession(i-1));
end

% second pass, smooth, single forward pass with the classical
% digital filter of Lyne and Hollick (1979) but applied to the Master
% recession rather than to total flow

quickrecessionflow=masterrecession; 

for i = 2:length(strflow)
    quickrecessionflow(i)=max(0,k*quickrecessionflow(i-1)+(masterrecession(i)-masterrecession(i-1))*(1+k)/2);
end

baseq=masterrecession-quickrecessionflow; 


%% References

% Duncan, H. P.: Baseflow separation  A practical approach, Journal of Hydrology, 575, 308313, https://doi.org/10.1016/j.jhydrol.2019.05.040, 2019.
% Nathan, R. J. and McMahon, T. A.: Evaluation of automated techniques for base flow and recession analyses, Water Resources Research, 26, 14651473, https://doi.org/10.1029/WR026i007p01465, 1990.
% Lyne, V. and Hollick, M.: Stochastic Time-Variable Rainfall-Runoff Modeling, Institution of Engineers Australia National Conference, 8992, 1979.

