%% Function for event separation for the upstream and the tributary gauges 
%Contact:
% Sergiy Vorogushyn and Bjrn Guse
% sergiy.vorogushyn@gfz-potsdam.de and bjrn.guse@gfz-potsdam.de
% GFZ German Research Centre for Geosciences, Hydrology Section, Telegrafenberg, Potsdam, Germany

function [Qa, Qa_series, QbaseS, QbaseE, Qpeak, Qdur, QindSdate, QindEdate, x_centroid, y_centroid, ytotal_centroid, QindSdate2, QindEdate2, Qvol, EVENTMAT] = Event_derivation_upstream_tributary_gauges(Q, QindSdate2_down, QindEdate2_down)
% extract baseflow at the beginning of the flood events corresponding to AMS peaks
%
% For flood event separation the Gradient-Methodology is used after Bastian
% Klein (2009) - Dissertation at the University of Bochum.

%
% input:
% Q: data series file with 4 columns in that order: day, month, year, discharge
%
% output:
% Qa: annual maximum discharge series or seasonal maximum discharge series
% Qa_series: array with discharges per hydological year
% QbaseS: baseflow series start for all extracted annual maxima
% Qbase_date: array containing the date of the event start (baseflow)
% corresponding to the AMS event

[m n] = size(Q);
if n ~= 4
    [m n] = size(Q');
    if n ~= 4
        error('data series must have 4 columns (day, month, year, Q)!');
    else
        Q = Q';
    end
end

meanQ = mean(Q(:,4));

% build gradients
% Original version of Klein(2009) computes gradient as the mean difference
% between two discharge values 2 days apart
gradient (1) = 0;
gradient(2:length(Q(:,4))-1) = (Q(3:end,4) - Q(1:end-2, 4))/2;
gradient(length(Q(:,4))) = 0;
gradient = gradient';

% Modified version: computes simple difference between two consecutive
% days (seems to work better).

gradient2 (1) = 0;
gradient2(2:length(Q(:,4))) = (Q(2:end,4) - Q(1:end-1, 4));
gradient2(length(Q(:,4))) = 0;
gradient2 = gradient2';

grad2size = size(gradient2);

% find the beginning of the hydrological year (1.11.) having complete series
ind = find(Q(:,1) == 1 & Q(:,2) == 11);

Q = Q(ind(1):size(Q(:,1)),:);

% Initialize variables and set flags
n_years = length(ind); % Number of years in the matrix
Qa = zeros(n_years,2); % 
QbaseS = zeros(n_years,4);
QbaseE = zeros(n_years,4);
Qdur = zeros(n_years,2);
QindEdate = zeros(n_years,1);
QindSdate = zeros(n_years,1);
QindEdate2 = zeros(n_years,1);
QindSdate2 = zeros(n_years,1);
Qpeak  = zeros(n_years,4);
y_centroid=zeros(n_years,1);
x_centroid=zeros(n_years,1);
ytotal_centroid=zeros(n_years,1);
Qvol=zeros(n_years,1);

Qa_series = zeros(367,n_years);
grad_series = zeros(367,n_years);
grad_series2 = zeros(367,n_years);
z = 0;
z2 = 0;

startye = 1;
endye = n_years-1;

EVENTMAT = zeros(100, endye-startye+1);

for i = startye:endye
    % sort discharge series along the years
    % sort corresponding gradients
    
    Estart_now = QindSdate2_down(i);
    Eend_now = QindEdate2_down(i);
    
    eventdur = Eend_now-Estart_now+2;
    
    EVENTMAT(1,i) = Q(Estart_now,3);
    EVENTMAT(2:eventdur,i) = Q(Estart_now:Eend_now,4);
    
    if i < (n_years-1)
       
        Qa(i,1) = Q(ind(i+1),3); % Year in the first column of Qa
        Q_jahr = Q(ind(i):ind(i+1)-1,4); % annual discharge time series
        Qa_series(2:length(Q_jahr)+1,i) = Q_jahr;
        grad_series(1:length(Q_jahr), i) = gradient(ind(i):ind(i+1)-1);
        grad_series2(1:length(Q_jahr), i) = gradient2(ind(i):ind(i+1)-1);
    
    else
              
        Qa(i,1) = Q(end,3);
        Q_jahr = Q(ind(i):end,4);
        Qa_series(2:length(Q_jahr)+1,i) = Q_jahr;
        grad_series(1:length(Q_jahr),i) = gradient(ind(i):end);
        grad_series2(1:length(Q_jahr),i) = gradient2(ind(i):end);
    end
    Qa_series(1,i) = Qa(i,1);
    Qa_series(end:end) = 0;
end    
    Qa_series (368,:) = 0; % matrix with 368 rows and n-years columns
    
% get AMS values and sort them along the years
    [ams,amsdate] = Annual_maxima_series(Q(:,1), Q(:,2), Q(:,3), Q(:,4));
         Qa(:,2) = ams; % second column: AMS
         % amsdate as date with three column
   
% Gradient threshold (change in discharge between 2 consecutive days)
% empirical gradient for total gradient series
% empirical threshold for gradient: 90% percentile meaning that 10% are
% higher
    [f,x] = ecdf(gradient2);
    Gthr = x(find(f>=0.9,1)); %take 90th percentile of the empirical distribution. Seems to work fine for nearly all gauges.
    Gthr2 = x(find(f>=0.1,1)); %take 10th percentile of the empirical distribution. Seems to work fine for nearly all gauges.
    % Since the first value is used in the find comment, also for Gthr2 >
    % is selected instead of <.
    
% Number of days with gradient below the threshold Gthr in order to detect
% baseflow
    TW =  7;
    
% Number of days for two peaks to be considered independent (Concentration
% time?), Klein (2009), S.110
    DT = 7;
    
% TLM (time window for local minimum) - number of days prior to AMS peak in
% which the local minimum is taken if the search for starting point of the
% hydrograph based on the gradient threshold did not deliver the result and
% the local minimum between two independent peaks is farther away than TLM
    TLM = 40;

% counter for days in the current year
    ndays = 0;
    j = 1;
       
for i = startye:endye    
    
    %   extract associated flood events
    Q_jahr = Qa_series(2:find(Qa_series(:,i)==0)-1, i); % Discharge as current year
     max_Q_jahr = max(Q_jahr);
     
    % flood event interval
    event_S = QindSdate2_down(i);
    event_E = QindEdate2_down(i);
    
   if (event_S>ndays)
       
       if (event_E<(ndays+365))
    
    % to avoid peaks before the beginning of the year
    event = Q(event_S:event_E,:);
    
    % first day with increasing discharge
    event_count = 1:size(event,1);
    
    event_gradient = gradient2(event_S:event_E);
    event_gradient_pos = event_gradient>0;
    event_gradS = event_count(event_gradient_pos)-1;
    
    % avoid an index of zero
    event_gradS = event_gradS(event_gradS>0);
    
    event_gradient_neg = event_gradient<0;
    event_gradE = event_count(event_gradient_neg);
    
    event_grad_posneg = sum(event_gradient_pos)*sum(event_gradient_neg);
    
    event_orig = event;
    
    if (event_grad_posneg>0) % check if there are both positive and negative gradients
           
        % empty check
       size_gradS = size(event_gradS);
       size_gradE = size(event_gradE);
        
       size_gradquo = size_gradS(1)*size_gradE(1);
        
       if (size_gradquo>0)
        
          if (event_gradE(end) > event_gradS(1)+1) %check if there is an event
    
             event = event_orig(event_gradS(1):event_gradE(end),:);
     
             % timing of maximum within the event
             max_event = max(event(:,4));
        
             % indQ - index of the cell with maximum annual discharge in the daily
             % annual series
             % indQ = find(Q_jahr == Qa(i,2));
             indQ = find(Q_jahr == max_event);
    
             if (length(indQ)>1)
                 indQ_check = indQ+ndays;
                 indQx = indQ_check>event_S;
                 indQ = indQ(indQx);
                 indQ_check = indQ+ndays;
                 indQx2 = indQ_check<event_E;
                 indQ = indQ(indQx2);        
             end
    
             indQ = indQ(1);

%%%%%   FLOOD PEAK INDEPENDENCY %%%%%
 
% Apply criteria for independency of flood peaks after Klein(2009),
% modified from LAWA(1997) and Bacchi et al. (1992).

% Make initialization of indices
% [var]2 for variables after the peak
             k = 1;
             k2 = 1;
             ind_flag = false;
             ind_flag2 = false;
             loc_max_peak = ndays+indQ;
             loc_max_peak2 = loc_max_peak;
             loc_min = ndays+indQ-1;
             % local minimum after peak
             loc_min2 = ndays+indQ+1;
    
% indS and indS1 are two alternative indices for the starting point of a
% flood hydrograph. indS1 is determined according to the threshold gradient and time window.
% In case indS1 is not detected or is located further away back in time 
% comapred to the local minimum between two independent peaks (indS), the
% indS is used.

             indS = []; %local minimum between two independent peaks before AMS
             indE = []; %local minimum between two independent peaks after AMS
             indS1 = []; %local minimum within a TW window with gradient below Gthr before peak
             indE1 = []; %local minimum within a TW window with gradient below Gthr after peak
             indSsel = [];
             indEsel = [];
 
    % move backward in time starting from the AMS peak value
             indQcheck = ndays+ indQ;
    
             while (ind_flag == false) && (ndays+indQ-k-1 > 0)  % while independent event is not found

% search for local minima
                 if (gradient2(ndays+indQ-k) >= 0) && (gradient2(ndays+indQ-k-1) < 0)

% index of the local minimum
% assign the local minimum between two peaks
                       if Q(ndays+indQ-k-1, 4) < Q(loc_min,4)
                          loc_min = ndays+indQ-k-1;
                       end
                 end
% search for the local maximum      
                 if (gradient2(ndays+indQ-k) <= 0) && (gradient2(ndays+indQ-k-1) > 0)
% index of the local maximum          
                     loc_max = ndays+indQ-k-1;
% Check whether the local maximum can be regarded as a peak (and not just
% the product of discharge oscillation) (Klein, 2009; S.110)
                     if (Q(loc_min,4) <= 0.7*Q(loc_max,4)) && ...
                        (Q(loc_max,4) >= 0.2*Q(ndays+indQ, 4))

% Check whether the peaks are independent
% Klein combines criteria after LAWA(1997) and Bacchi et al.(1992)
                        if ((loc_max_peak - loc_max) > DT) ...
                           ||  (Q(loc_min,4) <= 0.3*Q(loc_max_peak, 4))
                
                           indS = loc_min;
                           ind_flag = true;                
                        end
% Remember the local maxima that was identified as a peak
                        loc_max_peak = loc_max;
% erase the previous local minimum between the peaks and start searching
% for a new local minimum
                        loc_min = ndays+indQ - 1; %-k?
                     end
                 end
                 k = k + 1;
               
                 if ((ndays+indQ+k+1) > grad2size(1))
                     ind_flag2 = true;
                     loc_min = grad2size(1);
                     indS = loc_min;
                     loc_max_peak = loc_max;
                 end
           end
    
        % move forward in time starting from the AMS peak value
           while (ind_flag2 == false) && (ndays+indQ+k2+1 > 0)  % while independent event is not found

% search for local minima
                if (gradient2(ndays+indQ+k2) <= 0) && (gradient2(ndays+indQ+k2+1) > 0)

% index of the local minimum
% assign the local minimum between two peaks
                     if Q(ndays+indQ+k2, 4) < Q(loc_min2,4)
                        loc_min2 = ndays+indQ+k2;
                     end
               end

                % search for the local maximum
               if (gradient2(ndays+indQ+k2) >= 0) && (gradient2(ndays+indQ+k2+1) < 0)
% index of the local maximum          
                  loc_max2 = ndays+indQ+k2;
% Check whether the local maximum can be regarded as a peak (and not just
% the product of discharge oscillation) (Klein, 2009; S.110)
                  if (Q(loc_min2,4) <= 0.7*Q(loc_max2,4)) && ...
                     (Q(loc_max2,4) >= 0.2*Q(ndays+indQ, 4))

% Check whether the peaks are independent
% Klein combines criteria after LAWA(1997) and Bacchi et al.(1992)
                      if ((loc_max2 - loc_max_peak2) > DT) ...
                          ||  (Q(loc_min2,4) <= 0.3*Q(loc_max_peak2, 4))
                
                          indE = loc_min2;
                          ind_flag2 = true;
                
                      end
% Remember the local maxima that was identified as a peak
                     loc_max_peak2 = loc_max2;
% erase the previous local minimum between the peaks and start searching
% for a new local minimum
                     loc_min2 = ndays+indQ + 1;
                  end
                end
       
               k2 = k2 + 1;
               
               if ((ndays+indQ+k2+1) > grad2size(1))
                  ind_flag2 = true;
                  loc_min2 = grad2size(1);
                  indE = loc_min2;
                  loc_max_peak2 = loc_max2;
               end
           end
             
          indE = event_E;
          indS = event_S;

    %%%% HYDROGRAPH SEPARATION %%%%
    
% Apply gradient for hydrograph separation

% Place and move a time window from the day of the annual maximum flow  
% Pick-up the indices of the start point of a hydrograph after TW
% consecutive days with the gradient below the threshold.

% Move backward in time.
          cons_days = 0;
          for k = 0:1:(ndays+indQ-indS-TW+1)
        % control if gradient is lower than threshold
        % counting of days below threshold
             if abs(gradient2(ndays+indQ-k)) <= Gthr
                cons_days = cons_days + 1;
             else
                cons_days = 0;
             end
             if cons_days == TW
           
           %select the index of minimum flow within the time window,
           %when the gradient is below threshold (indS1)
               [gg, jj] = min(Q(ndays+indQ-k : ndays+indQ-k+TW-1,4));
                indS1 = ndays+indQ-k+jj-1; 
               break %starting point is found, quit the loop
             end
           end
        
    % Move forward in time, approach2
            cons_days2 = 0;
           for k = 0:1:(ndays+indQ-indE-TW+1)
        % control if gradient is lower than threshold
        % counting of days below threshold
              if abs(gradient2(ndays+indQ+k)) >= Gthr2
                 cons_days2 = cons_days2 + 1;
              else
                 cons_days2 = 0;
              end
              if cons_days2 == TW
           
           %select the index of minimum flow within the time window,
           %when the gradient is below threshold (indS1)
                [gg, jj] = min(Q(ndays+indQ+k : ndays+indQ+k+TW-1,4));
                indE1 = ndays+indQ+k+jj-1; 
                break %starting point is found, quit the loop
              end
           end
    
    %%%% LOCAL MINIMUM WITHIN TLM DAYS %%%%
    
% if the starting point has not been identified based on the threshold
% criteria and selected window size (indS1 is empty) and the local minimum
% between two independent peaks is more than TLM days prior to the AMS peak
% take the local minimum within TLM days prior to the AMS value
          if isempty(indS1) && (ndays+indQ-indS > TLM) % for dillingen indS is empty
             [M,I] = min(Q(ndays+indQ-TLM+1 : ndays+indQ,4));
             indS = ndays+indQ-TLM+I;
             QbaseS(i,:) = Q(indS(1),:);

% Run the count of cases, in whcih the starting date has not been
% identified and TLM window needs to be applied
             z = z+1;
          else
             
% Check if the identified wave starting point according to the
% gradient criterion (indS1) was detected and it occurs closer to the peak,
% otherwise take the local minimum between two independent peaks (it is already within TLM window)

           if ~isempty(indS1) 
               if (indS1 > indS)
                   QbaseS(i,:) = Q(indS1,:);
               else
                   QbaseS(i,:) = Q(indS,:);
               end
           else
               QbaseS(i,:) = Q(indS,:);
           end
         end

% forward approach
      
% if the ending point has not been identified based on the threshold
% criteria and selected window size (indE is empty) and the local minimum
% between two independent peaks is more than TLM days after the AMS peak
% take the first local minimum after the AMS value that is below mean
% annual flow

        if isempty(indE1) && (indE-ndays-indQ > TLM)
           [M,I] = findpeaks(-Q(ndays+indQ : ndays+indQ+2*TLM-1,4));
        % mean Q jahr times 2 as threshold to avoid a too early end of the
        % event
           indE = ndays + indQ + I(find(-M <= mean(Q_jahr)*2,1))-1; 
        
           QbaseE(i,:) = Q(indE(1),:);

% Run the count of cases, in which the ending date has not been
% identified and TLM window needs to be applied
           z2 = z2+1;
                                                              else
       
% Check if the identified wave starting point according to the
% gradient criterion (indE1) was detected and it occurs closer to the AMS peak,
% otherwise take the local minimum between two independent peaks (it is already within TLM window)

        if ~isempty(indE1) 
            if (indE1 < indE)
                QbaseE(i,:) = Q(indE1,:);
            else
                QbaseE(i,:) = Q(indE,:);
            end
        else
            QbaseE(i,:) = Q(indE,:);
        end
    end

    
% Control plots
 
    figure(i);
    plot(Qa_series(2:end,i)); %daily flow series for each year
    
    hold on;
    % plot of starting point of event
    scatter(indS-ndays,Q(indS,4),'r'); %local minima between two independent peaks
    
    QindSdate(i,:) = indS-ndays;
   
    scatter(indE-ndays,Q(indE,4),'o'); %local minima between two independent peaks
   
    QindEdate(i,:) = indE-ndays;
        
        %scatter(indQ,Q(indQ,4),'k'); %peak

    if (indS1 ~= 0)
        hold on;
        scatter(indS1-ndays, Q(indS1,4),'b'); %local minima within a time window with gradient below threshold
    end
    
    if (indE1 ~= 0)
        hold on;
        scatter(indE1-ndays, Q(indE1,4),'o'); %local minima within a time window with gradient below threshold after the peak
    end
        
    hold on;
    
    % maximum value of event
    Qpeakevent = Q(indQ+ndays,4);
    
    %scatter(indQ,Qa(i,2),'k'); %mark the AMS peak
    scatter(indQ,Qpeakevent,'k'); %mark the AMS peak
    
    % save the date of the AMS peak
    Qpeak(i,:) = Q(indQ+ndays,:);
 
      
    %% CENTROID CALCULATION
      
    figure(i+n_years);
      
      % selected starting point
      %indSsel = (max(indS, event_S));
    indSsel = indS;
      
    if indS1 > 0
      indSsel = max(indS, indS1);
    end
      
      % year and duration of event      
    Qdur(i,1) = amsdate(i,3);
    qdur = indE-indSsel+1;
    Qdur(i,2) = qdur;
     
    if (qdur>2)
      
      % selection of the event
      peak=[];
      peak = Q(indSsel:indE,:); 
     
    % plot(peak(:,4))
     % difference in Q between start and end of peak
      q_diff = Q(indE,4)-Q(indSsel,4);
    
    % linear line from indE to indS
      qcf = q_diff/(qdur-1);
   
    %%%% Q baseflow estimation
      q_baseflow=[];
      q_baseflow(1) = Q(indSsel,4);
    
      for kp = 2:qdur            
            q_baseflow(kp) = q_baseflow(1)+qcf*(kp-1);            
      end
      
      % peakflow = totalflow-baseflow
      q_peakflow =transpose(peak(:,4))-q_baseflow;
       
       %%%% RISING LIMB REDUCTION
       % reduction of the peak length in the rising limb in the case that
       % the calculated baseflow exceeds the peak discharge in the rising
       % limb       
      q_peakflow_rise = [];
      q_peakflow_rise = q_peakflow(1:(indQ+ndays-indSsel+1));
       
      q_peakflow_rise_neg = q_peakflow_rise<-0.001;
              
      while sum(q_peakflow_rise_neg) > 0
       
        rise_count = 1:length(q_peakflow_rise);
       
        indSsel = max(rise_count(q_peakflow_rise_neg))+indSsel-1;
       
        % Update of event duration
        qdur = indE-indSsel+1;
        Qdur(i,2) = qdur;
    
        peak=[];
        peak = Q(indSsel:indE,:); 
        
        % Difference in time between event start and end
        q_diff = Q(indE,4)-Q(indSsel,4);
    
    % linear line from indE to indS
        qcf = q_diff/(qdur-1);
   
        q_baseflow=[];
        q_baseflow(1) = Q(indSsel,4);
    
        for kp = 2:qdur            
            q_baseflow(kp) = q_baseflow(1)+qcf*(kp-1);            
        end
       
        q_peakflow =transpose(peak(:,4))-q_baseflow;
        q_peakflow_rise = q_peakflow(1:(indQ+ndays-indSsel+1));      
        q_peakflow_rise_neg = q_peakflow_rise<-0.001;
      
     end
       
       %%%% FALLING LIMB REDUCTION
       % reduction of the peak length in the falling limb in the case that
       % the calculated baseflow exceeds the peak discharge in the falling
       % limb
      q_peakflow_fall = [];
      q_peakflow_fall = q_peakflow((indQ+ndays-indSsel+1):end);
       
      q_peakflow_fall_neg = q_peakflow_fall<-0.001;
              
     while sum(q_peakflow_fall_neg) > 0
       
        fall_count = 1:length(q_peakflow_fall);
       
        indEsel = min(fall_count(q_peakflow_fall_neg))+indQ-1+ndays;
       
        % Update of event duration
        qdur = indEsel-indSsel+1;
        Qdur(i,2) = qdur;
      
        peak=[];
        peak = Q(indSsel:indEsel,:);
        
        % Difference in time between event start and end
        q_diff = Q(indEsel,4)-Q(indSsel,4);
    
    % linear line from indE to indS
        qcf = q_diff/(qdur-1);
   
        q_baseflow=[];
        q_baseflow(1) = Q(indSsel,4);
    
        for kp = 2:qdur            
            q_baseflow(kp) = q_baseflow(1)+qcf*(kp-1);            
        end
       
        q_peakflow =transpose(peak(:,4))-q_baseflow;
        q_peakflow_fall = q_peakflow((indQ+ndays-indSsel+1):end);      
        q_peakflow_fall_neg = q_peakflow_fall<-0.001;

    end
       
       %%% SUM OF DAILY PEAKFLOW
       
    q_peakflow_sum=[];
       % sum of daily peakflow
    for kp2 = 2:qdur           
        q_peakflow_sum(kp2) = sum(q_peakflow(1:kp2));
    end
       
    stpoint = indSsel-ndays+1;
    if stpoint <1
         stpoint=1;
    end
    xstart=[];
    xstart(1:stpoint)=NaN;
     
    xpeak = xstart;
    xpeak(stpoint:stpoint+(qdur-1)) = peak(:,4);
     
    plot(xpeak)
     
    hold on;
         
    xbase = xstart;
    xbase(stpoint:stpoint+(qdur-1)) = q_baseflow;
     
    plot(xbase)
     
       % peak volume
    q_peakvol = q_peakflow_sum(end);
                       
    QindSdate2(i,:) = indSsel;
    QindEdate2(i,:) = indE;
      
    Qvol(i) = q_peakvol;
       
      
    %% Calculation of x-centroid
       
    if (qdur>2)
       
       % half of peak volume
       q_peakvol_half  = q_peakvol/2;
       
       % days before (1) and after (0) half of the peak volume
       qxthresh = q_peakvol_half > q_peakflow_sum;
       qxt = sum(qxthresh);
       
       qxquo = (q_peakvol_half-q_peakflow_sum(qxt))/(q_peakflow_sum(qxt+1)-q_peakflow_sum(qxt));
    
       % Time of centroid
       xcentroid = qxt + qxquo;
       
       %% Calculation of Y-Centroid
       eventtime = transpose(1:qdur);
    
       q_peakflow_sort_down = sort(q_peakflow, 'descend');
       q_peakflow_sort_up = sort(q_peakflow);
    
    % distance between ordered discharge
       q_peakflow_sort_dist = q_peakflow_sort_down(1:(qdur-1))- q_peakflow_sort_down(2:qdur);
       
       % cumulative value of ordered discharge
       % the lowest value is all days included
       % the highest value only the days itself
       q_peakflow_sort_dist_cum=[];
       
       for kp3 = 1:(qdur-1)
            q_peakflow_sort_dist_cum(kp3,1) = q_peakflow_sort_dist(kp3)*kp3;
       end
        
       q_peakflow_sort_dist_cum(qdur,1)=0;
        
        % total event volume
       q_peakflow_sort_dist_total = sum(q_peakflow_sort_dist_cum);
        
       % half of the event volume (in y-direction)
       q_peakflow_half = q_peakflow_sort_dist_total/2;
        
       % order in opposite direction
       q_peakflow_sort_dist_cum_rev = wrev(q_peakflow_sort_dist_cum);
        
       q_peakflow_sort_dist_sum=[];
       for qd = 1:(qdur)
           q_peakflow_sort_dist_sum(qd) = sum(q_peakflow_sort_dist_cum_rev(1:qd));
       end
        
            % values below the half of the peakflow
       qythresh = q_peakflow_half > q_peakflow_sort_dist_sum;
   
       % sum of days below half of the peakflow
       qyt = sum(qythresh);
    
    % ratio between two days to calculate the subdaily value of the
    % centroid
       qyquo = (q_peakflow_half-q_peakflow_sort_dist_sum(qyt))/(q_peakflow_sort_dist_sum(qyt+1)-q_peakflow_sort_dist_sum(qyt));
    
     % calculation of the ycentroid (without baseflow)
       ycentroid = q_peakflow_sort_up(qyt) + qyquo*(q_peakflow_sort_up(qyt+1)-q_peakflow_sort_up(qyt));
         
    % baseflow value at the time point of the y-centroid
       baseflow_xcentroid =  q_baseflow(1)+qcf*(xcentroid-1);
    
    % Y-centroid of the total discharge (incl baseflow)
       ycentroid_total = ycentroid + baseflow_xcentroid;
    
       scatter(xcentroid+indSsel-ndays, ycentroid_total);
         
       x_centroid(i) = xcentroid+indSsel-ndays;
       y_centroid(i) = ycentroid;
       ytotal_centroid(i) = ycentroid_total;
    
    else
       x_centroid(i) = 0;
       y_centroid(i) = 0;
       ytotal_centroid(i) = 0;    
   
    end
  
      
  end 
   
  else
    x_centroid(i) = 0;
    y_centroid(i) = 0;
    ytotal_centroid(i) = 0;   
       
    q_noevent = Q(event_S:event_E,:);
    qmax_noevent = max(q_noevent(:,4));
    
    qnoid = q_noevent(:,4)==qmax_noevent;
    
    q_noevent2 = q_noevent(qnoid,:);
    Qpeak(i,:) = q_noevent2(1,:);
    QindSdate(i,:) = indS-ndays;
    QindEdate(i,:) = indE-ndays;
    QindSdate2(i,:) = indSsel;
    QindEdate2(i,:) = indE;
    Qdur(i,1) = amsdate(i,3);
    qdur = indE-indSsel+1;
    Qdur(i,2) = qdur;
            
  end %if event check

 end % empty check
    
end % if pos + neg gradient check
    
       end 
       
   end
   
    ndays = ndays + length(Q_jahr);
        
    % How often the starting point has not been identified based on
    % threshold criteria
    z;
    z2;
   
end