function [size_res, perc_size, entropy,SNR]=compress_ratio_mopex(inputtype,methods,series)
%[size_res, perc_size, entropy,SNR,signals_struct]=compress_ratio_mopex(inputtype,methods,series)
%  test compression performance of various existing compression algorithms
%  on different types of data
%  note: calls arj.exe, 7z.exe, and wavpack.exe, which need to be installed
%  in the correct subfolders. This code has only been tested on windows 7 operating system. 
%  
% inputs:
%  inputtype - an integer that specifies the data to compress, either generated 
%              inside this function or supplied as 3rd argument
%  methods   - an vector of integers that specifies the compression methods
%              to use
%  series    - the externally supplied time series to compress, for methods
%              numbers -1 0 20 and 21, if supplied as uint8 type vectors or matrix, 
%              it is assumed that values have been quantized already,
%              otherwise, linear quantization over range is preformed. 
%
% outputs:
%  size_res  - M by N matrix with resulting file sizes in bytes, 
%               where M is the number of compression methods used and N is number of signals compressed 
%  perc_size - Same, but file size as percentage of original file size
%  entropy   - Entropy of quantized signals 
%  SNR       - Signal to noise ratio
% 
%     
% The inputtype and methods are coded with the following integers (see source code below for more details)
%
% 
% INPUT SIGNAL TYPES
% EXTERNAL_RAW=-1;
% EXTERNAL=0;
% CONSTANT=1;
% LINEAR=2;
% UNIFORM_WHITE=3;
% NORMAL_WHITE=4;
% SINE1=5;
% SINE100=6;
% LEAFQ=7;
% LEAFP=8;
% LEAF_ERR=9;
% LEAF_ERRDIFF=10;
% LEAFQMOD    =11;
% MULT_SERIES = 20;
% MULT_SERIES_RAW = 21;
%
% COMPRESSION METHODS:
% NONE        =1;
% ARJ         =2;
% WAV         =3;
% WAVPACK     =4;
% BMP         =5;
% JPG_LOSSLESS=6;
% JPG_50      =7;
% GIF         =8;
% HDF_NONE    =9;
% HDF_JPG     =10;
% HDF_RLE     =11;
% PPMD        =12;
% LZMA        =13;
% BZIP2       =14;
% PNG         =15;
% TIFF        =16;
% PNG_365     =17;
% SFX         =18;
%
%
%  This code is supplied as supplementary material with the purpose of making it easier to reproduce
%  the results presented in the following paper: 
%    Weijs, S. V.; van de Giesen, N. & Parlange, M. B. 
%    Data compression to define information content of hydrological time series 
%    Hydrology and Earth System Sciences Discussions, 2013, 10, 2029-2065
%    http://www.hydrol-earth-syst-sci-discuss.net/10/2029/2013/hessd-10-2029-2013.html
%    
%  The code is experimental and not optimized or commented extensively.
%  For questions, feel free to contact the first author: steven.weijs@epfl.ch 
%  Version 1.0; future updates might become available on: 
%   http://www.hydroinfotheory.net/compression_hess

%settings
filesize=50000;
outdir='output\';

%define input signal type
EXTERNAL_RAW=-1;
EXTERNAL=0;
CONSTANT=1;
LINEAR=2;
UNIFORM_WHITE=3;
NORMAL_WHITE=4;
SINE1=5;
SINE100=6;
LEAFQ=7;
LEAFP=8;
LEAF_ERR=9;
LEAF_ERRDIFF=10;
LEAFQMOD    =11;
MULT_SERIES = 20;
MULT_SERIES_RAW = 21;

%define compression methods
NONE        =1;
ARJ         =2;
WAV         =3;
WAVPACK     =4;
BMP         =5;
JPG_LOSSLESS=6;
JPG_50      =7;
GIF         =8;
HDF_NONE    =9;
HDF_JPG     =10;
HDF_RLE     =11;
PPMD        =12;
LZMA        =13;
BZIP2       =14;
PNG         =15;
TIFF        =16;
PNG_365     =17;
SFX         =18;

% columns of series to use (now all cols provided)
sr_cols=1:size(series,2);
%number of columns
nr_cols=length(sr_cols);

%initialization
size_res=zeros(length(methods),nr_cols);
perc_size=size_res;
entropy(nr_cols)=0;
SNR(nr_cols)=0;

multicol=0;
%inputtype=NORMAL_WHITE
%inputtype=SINE;
%inputtype=LEAFQ;



switch inputtype
    case EXTERNAL_RAW
        hydro_series=uint8(series);
        filesize=size(series,1);
    case EXTERNAL
        hydro_series=series;
        filesize=size(series,1);
    case LEAFQ
        % input data
        load leaf4888d
        hydro_series=Qobs';
        hydro_series=Qobs';
        %do whole years only
        %hydro_series=hydro_series(1:end-mod(length(hydro_series),365));
        %filesize=size(Qobs,1);
        filesize=length(hydro_series);
    case LEAFQMOD
        % input data
        load leaf4888d
        hydro_series=Qd';
        %do whole years only
        %hydro_series=hydro_series(1:end-mod(length(hydro_series),365));
        %filesize=size(Qobs,1);
        filesize=length(hydro_series);        
    case LEAFP
        % input data
        load leaf4888d
        hydro_series=P';
        %do whole years only
        %hydro_series=hydro_series(1:end-mod(length(hydro_series),365));
        %filesize=size(P,1);
        filesize=length(hydro_series);
    case LEAF_ERR
        hydro_series=read_binary('errorRR2.bin');
        filesize=length(hydro_series);
        hydro_series=uint8(hydro_series);
    case LEAF_ERRDIFF
        hydro_series=read_binary('errorRR3.bin');
        filesize=length(hydro_series);
        hydro_series=uint8(hydro_series);
    case UNIFORM_WHITE
        hydro_series=rand(1,filesize);
    case NORMAL_WHITE
        hydro_series=randn(1,filesize);
    case CONSTANT
        hydro_series=ones(1,filesize);
    case LINEAR
        hydro_series=1:filesize;
    case SINE1
        t=1:filesize;
        %period in samples P
        period=filesize;
        hydro_series=sind(360*t/period);
    case SINE100
        t=1:filesize;
        %period in samples P
        period=filesize/100;
        hydro_series=sind(360*t/period);
    case MULT_SERIES
        %we have multiple series at same time
        hydro_series=series;
        filesize=size(series,1);
        multicol=1;
    case MULT_SERIES_RAW
        %we have multiple series at same time
        hydro_series=uint8(series);
        filesize=size(series,1);
        multicol=1;
        
end

% loop over column series
for sr=sr_cols
    hydro_series=series(:,sr);
    %% various derived series
    %scale between 0 and 1
    min_val=double(min(hydro_series));
    max_val=double(max(hydro_series));
    range_val=max_val-min_val;
    if (range_val==0)
        range_val=1;
    end
    scaled_series=(double(hydro_series-min_val)./range_val);

    %do quantization only for series that are not yet quantized
    if isa(hydro_series,'uint8')
        quantized_1byte=hydro_series;
    else
        % and quantize 1byte
        %quantized_1byte=uint8(round(scaled_series*255)); 
        quantized_1byte=uint8(floor(scaled_series*255));
    end
    %figure;
    %plot(scaled_series);
    distr=hist(double(quantized_1byte),256);
    entropy(sr)=histogram_entropy(distr);
    %convert to -1..1
    wav_series=-0.9999+(scaled_series*1.9998);
    %filename_out(size(methods,2))=0

    err=   (double(quantized_1byte)./255) -scaled_series ;
    sigma_noise=std(err);
    sigma_signal=std(scaled_series);
    SNR(sr)=sigma_signal/sigma_noise;
    
    if multicol
        inputtype=sr;
    end
        
    %% compress with different algorithms
    for compression_method=methods;
        filename_out=sprintf('sig%im%i',inputtype,compression_method);
        %filename_out_ext=('sig%im%i.%s',inputtype,compression_method,ext(compression_method));

        switch compression_method
            case NONE
                %save as ascii
                series_ascii=char(quantized_1byte);
                min(quantized_1byte);
                max(quantized_1byte);

                fid = fopen([outdir filename_out '.txt'],'w');
                fwrite(fid, series_ascii);
                fclose(fid);
            case WAV
                %save as wav
                wavwrite(wav_series,8000,8,[outdir filename_out]);
                %wavwrite(quantized_1byte,8000,8,'serie.wav');

            %save as pictures
            case BMP
               imwrite(quantized_1byte,[outdir filename_out '.bmp'],'bmp');
            case JPG_LOSSLESS
                imwrite(quantized_1byte,[outdir filename_out '.jpg'],'jpg','mode','lossless');
            case JPG_50
                imwrite(quantized_1byte,[outdir filename_out '.jpg'],'jpg','mode','lossy','Quality',50);
            case GIF
                imwrite(quantized_1byte,[outdir filename_out '.gif'],'gif');
            case HDF_NONE
                imwrite(quantized_1byte,[outdir filename_out '.hdf'],'hdf','compression','none');
            case HDF_RLE
                imwrite(quantized_1byte,[outdir filename_out '.hdf'],'hdf','compression','rle');
            case HDF_JPG
                imwrite(quantized_1byte,[outdir filename_out '.hdf'],'hdf','compression','jpeg','quality',100);    
            case PNG
                imwrite(quantized_1byte,[outdir filename_out '.png'],'png'); 
            case PNG_365
                %add ones to make length multiple of 365
                add_length=365-rem(length(quantized_1byte),365);
                rawdata=[quantized_1byte; uint8(255*ones(add_length,1))];
                %reshape to width=365
                rawdata=reshape(rawdata,365,[]);
                imwrite(rawdata,[outdir filename_out '.png'],'png');
                %save as PNG
            case TIFF
                imwrite(quantized_1byte,[outdir filename_out '.tif'],'tiff','compression','packbits'); 


            %other conpression algorithms
            case ARJ
                %compress
                filename_in=sprintf('sig%im%i',inputtype,NONE);
                [~,~]=dos([pwd '\arj\arj.exe a -jm1 ' outdir filename_out ' ' outdir filename_in '.txt']);
            case WAVPACK
                filename_in=sprintf('sig%im%i',inputtype,WAV);
                [~,~]=dos(['del ' outdir filename_out '.wv']);
                [~,~]=dos([pwd '\wavpack\wavpack.exe ' outdir filename_in '.wav ' outdir filename_out '.wv' ]);
            case PPMD
                filename_in=sprintf('sig%im%i',inputtype,NONE);
                [~,~]=dos([pwd '\7-zip\7z.exe a -bd -m0=ppmd:o=32 ' outdir filename_out ' ' outdir filename_in '.txt']);
            case LZMA
                filename_in=sprintf('sig%im%i',inputtype,NONE);
                [~,~]=dos([pwd '\7-zip\7z.exe a -bd -m0=lzma:a=2 ' outdir filename_out ' ' outdir filename_in '.txt'])  ;  
            case BZIP2   
                filename_in=sprintf('sig%im%i',inputtype,NONE);
                [~,~]=dos([pwd '\7-zip\7z.exe a -bd -m0=bzip2:pass=9 ' outdir filename_out ' ' outdir filename_in '.txt']);
            case SFX
                filename_in=sprintf('sig%im%i',inputtype,NONE);
                [~,~]=dos([pwd '\7-zip\7z.exe a -bd -sfx ' outdir filename_out ' ' outdir filename_in '.txt']);


        end % end switch
    end %end loop methods
end % end loop series    
    %test data quality
    %image_array=imread([outdir 'serie.gif']);
    
%% test resulting file_sizes
s_nr=0;
for sr=sr_cols
  
    m_nr=0;
    s_nr=s_nr+1;
    for cm=methods
        m_nr=m_nr+1;
        if multicol
            filename_info=sprintf('sig%im%i',s_nr,cm);
        else
            filename_info=sprintf('sig%im%i',inputtype,cm);
        end  
        
        file_info=dir(['output\' filename_info '.*']);
        size_res(m_nr,s_nr)= file_info(1,1).bytes;
        perc_size(m_nr,s_nr)=size_res(m_nr,s_nr)/filesize;
    end
end
